--[[----------------------------------------------------------------------------

    ReverseGeocode is a plugin for Adobe Lightroom that fills location metadata
     based on GPS coordinates and information from OpenStreetMap Nominatim API.
    
    Copyright (C) 2019  Diego del Pozo
    
    Thanks to Michael Bungenstock for his Teekesselchen open plugin. Checking its
    code made me try to develop a new plugin   
 
--------------------------------------------------------------------------------

ReverseGeocodeDialog.lua

This is the settings dialog.

------------------------------------------------------------------------------]]

local LrApplication = import "LrApplication"
local LrBinding = import "LrBinding"
local LrDialogs = import "LrDialogs"
local LrFunctionContext = import "LrFunctionContext"
local LrView = import "LrView"
local LrTasks = import "LrTasks"
local LrDate = import "LrDate"
local LrHttp = import "LrHttp"

require "ReverseGeocode"
require "Updater"

local license = LOC "$$$/ReverseGeocode/SWLicense=Copyright 2019 Diego del Pozo \n\nRedistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:\n\n  1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.\n  2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution."

local license2 = LOC "$$$/ReverseGeocode/SWLicense2=THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS \"AS IS\" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."

local licenseMAP = LOC "$$$/ReverseGeocode/MapLicense=Map data is provided by ^C OpenStreetMap contributors and licensed under the Open Data Commons Open Database License (ODbL) by the OpenStreetMap Foundation (OSMF). More information at <https://www.openstreetmap.org/copyright>"

local Openstreetmaps = 'nominatim.openstreetmap.org'

local function validateEmail (email)
	if email == nil or email =="" then
		return true
	else 
		if email:match("[A-Za-z0-9%.%%%+%-]+@[A-Za-z0-9%.%%%+%-]+%.%w%w%w?%w?") then
			return true
		else
			return false
		end
	end
end




local function showReverseGeocodeDialog()
	
	LrTasks.startAsyncTask( function()
		LrFunctionContext.callWithContext("myDialog", function(context)
			local revGeo = ReverseGeocode.new(context)
			local f = LrView.osFactory()
			local p = LrBinding.makePropertyTable(context)
			local configuration = _G.configuration
			local lrVersion = LrApplication.versionTable()
			local lrMajor = lrVersion.major
			local lrMinor = lrVersion.minor
			local supportsFlag = lrMajor >= 4
			
			configuration.copyTo(p)
			p.useFlag = p.useFlag and supportsFlag

			local contents = f:tab_view  {
				bind_to_object = p,
				width = 600,
				-- 1st tab
				f:tab_view_item {
					title = LOC "$$$/ReverseGeocode/DialogSummaryTabTitle=Summary",
					identifier = "SummaryTab",
					f:column {
						fill_horizontal = 1,
						spacing = f:label_spacing(),
						f:row {
							fill_vertical = 1,
							spacing = 0,
							f:static_text {
								title = LOC "$$$/ReverseGeocode/Process=I will check",
							},
							f:static_text {
								title = " ",
							},
							f:static_text {
								font = "<system/bold>",
								title = "" .. revGeo.total,
							},
							f:static_text {
								title = " ",
							},
							f:static_text {
								title = LOC "$$$/ReverseGeocode/Photos=photos",
							},
							f:static_text {
								title = ".",
							},
						},
	
						f:spacer {
							height = 20,						
						},
						
						f:static_text {
							title = LOC "$$$/ReverseGeocode/SummaryFields=Information gathered will be added to the following fields:",
							visible = LrView.bind("settingISOCountry") or LrView.bind("settingCountry") or LrView.bind("settingState") or LrView.bind("settingCity") or LrView.bind("settingLocation"),
						},
								
						f:static_text {
							title = LOC "$$$/ReverseGeocode/Bullet=^B " .. LOC "$$$/ReverseGeocode/settingISOCountry=ISO Country Code",
							visible = LrView.bind("settingISOCountry"),
						},
						f:static_text {
							title = LOC "$$$/ReverseGeocode/Bullet=^B " .. LOC "$$$/ReverseGeocode/settingCountry=Country",
							visible = LrView.bind("settingCountry"),
						},
						f:static_text {
							title = LOC "$$$/ReverseGeocode/Bullet=^B " .. LOC "$$$/ReverseGeocode/settingState=State/Province",
							visible = LrView.bind("settingState"),
						},
						f:static_text {
							title = LOC "$$$/ReverseGeocode/Bullet=^B " .. LOC "$$$/ReverseGeocode/settingCity=City",
							visible = LrView.bind("settingCity"),
						},
						f:static_text {
							title = LOC "$$$/ReverseGeocode/Bullet=^B " .. LOC "$$$/ReverseGeocode/settingLocation=Secondary Location",
							visible = LrView.bind("settingLocation"),
						},
						
						f:spacer {
							height = 20,						
						},
						
						f:static_text {
							font = "<system/bold>",
							title = LOC "$$$/ReverseGeocode/SummaryOverwrite=Current information will be owerwritten!",
							visible = LrView.bind("settingOverwrite"),
						},

						f:spacer {
							height = 40,						
						},


						f:push_button {
							title = LOC "$$$/ReverseGeocode/resetDefault=Reset to Defaults",
							align = "right",
							action = function(button)
								configuration.copyDefaultsTo(p)
								p.useFlag = p.useFlag and supportsFlag
							end,
						},
					},
					f:column {
						fill_horizontal = 1,
						place_vertical = 1,
						f:static_text {
							fill_horizontal = 1,
								alignment = "right",
								text_color = "blue",
								title = LOC "$$$/ReverseGeocode/Help=Help",
								tooltip = LOC "$$$/ReverseGeocode/DialogHelp=Click on me to open help in browser.",
								font = "<system/small>",
								mouse_down = function(o)
									LrHttp.openUrlInBrowser("http://www.burgaleses.es/reversegeocode.php#help")
								end
							},
						},
				},
				-- 2nd tab
				f:tab_view_item {
					title = LOC "$$$/ReverseGeocode/DialogSettingsTabTitle=Settings",
					identifier = "SettingsTab",
					f:column {
						fill_horizontal = 1,
						spacing = f:control_spacing(),
						--
						-- Options
						--
						f:group_box {
							title = LOC "$$$/ReverseGeocode/Options=Options",
							fill_horizontal = 1,
							spacing = f:control_spacing(),
							
							f:row {
								f:column {
							
									fill_horizontal = 0.75,
						
								
									f:row {
										fill_vertical = 1,
										spacing = 0,

										f:static_text {
											title = LOC "$$$/ReverseGeocode/DialogSettingOptions=Select the fields that will be filled with data from Internet",
											alignment = "right",
										},
									},

									f:checkbox {
										title = LOC "$$$/ReverseGeocode/settingISOCountry=ISO Country Code",
										value = LrView.bind("settingISOCountry"),
										enabled = true,
									},
							
									f:checkbox {
										title = LOC "$$$/ReverseGeocode/settingCountry=Country",
										value = LrView.bind("settingCountry"),
										enabled = true,
									},
									f:checkbox {
										title = LOC "$$$/ReverseGeocode/settingState=State/Province",
										value = LrView.bind("settingState"),
										enabled = true,
									},
									f:checkbox {
										title = LOC "$$$/ReverseGeocode/settingCity=City",
										value = LrView.bind("settingCity"),
										enabled = true,
									},
									f:checkbox {
										title = LOC "$$$/ReverseGeocode/settingLocation=Secondary Location",
										value = LrView.bind("settingLocation"),
										enabled = true,
									},	
								},
							
								f:column {
							
									fill_horizontal = 0.25,
							
									f:checkbox {
											title = LOC "$$$/ReverseGeocode/settingOverwrite=Overwrite",
											value = LrView.bind("settingOverwrite"),
											enabled = true,
										},
							
								},
							
							},
							
						},			

						f:group_box {
							title = LOC "$$$/ReverseGeocode/DialogNominatim=Nominatim Settings",
							fill_horizontal = 1,
							spacing = f:control_spacing(),
							f:row {
								f:static_text {
									title = LOC "$$$/ReverseGeocode/NominatimURL=Nominatim Address",
								},
								f:spacer {
									width = 2,
								},
								
								f:edit_field {
									value = LrView.bind("settingNominatimHost"),
									width_in_chars = 40,
--									validate = revGeo.check_URL,
								},
							},

							f:row {
								f:static_text {
									title = LOC "$$$/ReverseGeocode/UserEmail=E-mail",
									tooltip = LOC "$$$/ReverseGeocode/WhyEmail=Nominatim API requests a user email for large number of requests.",
								},
								
								f:spacer {
									width = 2,
								},
								
								f:edit_field {
									value = LrView.bind("settingEmail"),
									placeholder_string = LOC "$$$/ReverseGeocode/EnterEmail=Enter your email",
									tooltip = LOC "$$$/ReverseGeocode/WhyEmail=Nominatim API requests a user email for large number of requests.",
									width_in_chars = 40,
								},
								
							},

							f:row {
								f:static_text {
									title = LOC "$$$/ReverseGeocode/NomitatimTime=Time between queries",
									tooltip = LOC "$$$/ReverseGeocode/WhyTime=Openstreet Nominatim API imposes a minimum of 1 second between requests.",
									},

								f:edit_field {
									value = LrView.bind("settingNominatimTime"),
									tooltip = LOC "$$$/ReverseGeocode/WhyTime=Openstreet Nominatim API imposes a minimum of 1 second between requests.",
									precision = 1,
									increment = 0.1,
									large_increment = 1,
									width_in_chars = 3,
	--									validate = revGeo.check_time,
									},
								f:spacer {
									width = 2,
								},
								
								f:static_text {
									title = LOC "$$$/ReverseGeocode/Seconds=Seconds",
								},
							},								

							f:row {
								f:static_text {
									title = LOC "$$$/ReverseGeocode/Language=Language",
									tooltip = LOC "$$$/ReverseGeocode/WhyLanguage=RFC2616 language code for Nominatim results.\nUnrecognised languages will show results in English.",
									},

								f:edit_field {
									value = LrView.bind("settingLanguage"),
									tooltip = LOC "$$$/ReverseGeocode/WhyLanguage=RFC2616 language code for Nominatim results.\nUnrecognised languages will show results in English.",
	--									validate = revGeo.check_language,
									},
							},								




						},				
					},
					f:column {
						fill_horizontal = 1,
						place_vertical = 1,
						f:static_text {
							fill_horizontal = 1,
								alignment = "right",
								text_color = "blue",
								title = LOC "$$$/ReverseGeocode/Help=Help",
								tooltip = LOC "$$$/ReverseGeocode/DialogHelp=Click on me to open help in browser.",
								font = "<system/small>",
								mouse_down = function(o)
									LrHttp.openUrlInBrowser("http://www.burgaleses.es/reversegeocode.php#help")
								end
							},
						},
					},

				-- 3rd tab
				f:tab_view_item {
					title = LOC "$$$/ReverseGeocode/DialogAboutTabTitle=About",
					identifier = "AboutTab",
					f:column {
						fill_horizontal = 1,
						spacing = f:control_spacing(),
						
						f:row {
							fill_vertical = 1,
							spacing = 0,
							f:static_text {
								title = LOC "$$$/ReverseGeocode/PluginName=Reverse Geocode",
							},
							f:static_text {
								title = " v",
							},
							f:static_text {
								title = "" .. _G.CURRENT_VERSION,
							},
						},
						
						f:static_text {
							title = "Copyright © 2019  Diego del Pozo",
						},
						
						f:row {
							fill_vertical = 1,
							spacing = 0,
							f:static_text {
								title = LOC "$$$/ReverseGeocode/Contact=Contact",
							},
							f:static_text {
								title = ": ",
							},
							f:static_text {
								title = _G.myEmail,
							},
						},

					f:scrolled_view {
						height = 200,
						width = 550,
						horizontal_scroller = true,
						vertical_scroller = true,
						margin = 1,
						
						f:static_text {
							font = { name = "Helvetica Light", size = 10},
							alignment = "left",
							selectable = true,
							width_in_chars = 62,
							height_in_lines = -1,
							title = license,
							},
						f:static_text {
							font = { name = "Helvetica Light", size = 10},
							alignment = "left",
							selectable = true,
							width_in_chars = 62,
							height_in_lines = -1,
							title = license2,
							},
					},
						f:spacer {},
					f:scrolled_view {
						height = 30,
						width = 550,
						horizontal_scroller = true,
						vertical_scroller = true,
						margin = 1,
						
						f:static_text {
							font = { name = "Helvetica Light", size = 10},
							alignment = "left",
							selectable = true,
--							truncation = "tail",
							width_in_chars = 62,
							height_in_lines = -1,
							--enabled = false,
							title = licenseMAP,
							},
					},
						f:spacer {},
						
					f:group_box {
						title = LOC "$$$/ReverseGeocode/Updates=Updates",
						fill_horizontal = 1,
						spacing = f:control_spacing(),
					f:checkbox {
						title = LOC "$$$/ReverseGeocode/UpdatesTitle=Automatically check for updates",
						value = LrView.bind("checkForUpdates"),
					},
					f:push_button {
						title = LOC "$$$/ReverseGeocode/UpdatesButton=Check for updates now",
						alignment = "center",
						action = function(button)
							LrTasks.startAsyncTask( function()
								local u = Updater.new()
								local version=u.getVersion()
								if version then
									if version > _G.CURRENT_VERSION then
										local result = LrDialogs.confirm(LOC("$$$/ReverseGeocode/UpdatesAction=A new version is available (^1)",version), LOC "$$$/ReverseGeocode/UpdatesAction2=Select Update to open download web page in browser",LOC "$$$/ReverseGeocode/Update=Update")
										if result == "ok" then
											LrHttp.openUrlInBrowser(u.getUrl())
										end
									else
										LrDialogs.message(LOC "$$$/ReverseGeocode/UpdatesLastVersion=You are using the latest version.", LOC "$$$/ReverseGeocode/UpdatesNoUpdate=No update is necessary.", "info")
									end
								else
									LrDialogs.message(LOC "$$$/ReverseGeocode/UpdatesNoAnswer=Updates host not responding. Please check your internet connection.")
								end
							end)
						end,
					},
					},
					f:spacer {},
						f:group_box {
							title = LOC "$$$/ReverseGeocode/Debugging=Debugging",
							fill_horizontal = 1,
							spacing = f:control_spacing(),
							f:checkbox {
								title = LOC "$$$/ReverseGeocode/Logging=Logging",
								value = LrView.bind("activateLogging"),
							},
						},
					},
					f:column {
						fill_horizontal = 1,
						place_vertical = 1,
						f:static_text {
							fill_horizontal = 1,
								alignment = "right",
								title = LOC "$$$/ReverseGeocode/Help=Help",
								tooltip = LOC "$$$/ReverseGeocode/DialogHelp=Click on me to open help in browser.",
								font = "<system/small>",
								mouse_down = function(o)
									LrHttp.openUrlInBrowser("http://www.burgaleses.es/reversegeocode.php#help")
								end
							},
						},
				},
			}
			
			local result = LrDialogs.presentModalDialog({
				title = LOC "$$$/ReverseGeocode/DialogTitle=Reverse Geocode by Diego del Pozo",
				contents = contents,
				actionVerb = LOC "$$$/ReverseGeocode/DialogOK=Start",
				otherVerb = LOC "$$$/ReverseGeocode/DialogSavePref=Save preferences",
			})
			-- check user action
			if result == "other" then
				-- do not check the input, just save it
				configuration.copyFrom(p)
				configuration.write()
			else
				if result == "ok" then
					
					local msg 
					local errors = false

					-- do some error checking
					
					if string.match(p.settingNominatimHost, Openstreetmaps) ~= nil and p.settingNominatimTime < 1 then
						msg = LOC "$$$/ReverseGeocode/WhyTime=Openstreet Nominatim API imposes a minimum of 1 second between requests."
						errors = true
					end
						
					if not validateEmail(p.settingEmail) then
						msg = LOC "$$$/ReverseGeocode/ErrorEmail=Please enter a correct email address"
						errors = true 
					
					end
				
					-- show the dialog
					if errors then
						LrDialogs.message(LOC "$$$/ReverseGeocode/DialogErrorStart=Something is wrong with your settings.", msg)
					end
					configuration.copyFrom(p)
					configuration.write()
					-- is everything fine? then kick off
					if not errors then
						local startTime = LrDate.currentTime()
						revGeo.GPSInfo(configuration.settings)
						if revGeo.error then
							LrDialogs.showError(revGeo.errorString)
						else 					
							local timeAux = LrDate.currentTime() - startTime
							local hours = math.floor(timeAux / 3600)
							timeAux = timeAux - (3600 * hours)
							local minutes = math.floor(timeAux / 60)
							local seconds = timeAux - (60 * minutes) 
							local mytime = string.format("%.2d:%.2d:%02.3f", hours, minutes, seconds)
							local infoStr = LOC("$$$/ReverseGeocode/DialogDuration=Elapsed time : ^1", mytime)
							local messageStr = LOC("$$$/ReverseGeocode/DialogResult=Processed ^1 photos, ^2 bypassed, ^3 updated, ^4 no new information found", revGeo.total, revGeo.skipped, revGeo.found, revGeo.unknown) 

							LrDialogs.message(messageStr, infoStr, "info")
						end
					end
				end			  
			end
		end)
	end)
end

showReverseGeocodeDialog()