--[[----------------------------------------------------------------------------

    ReverseGeocode is a plugin for Adobe Lightroom that fills location metadata
     based on GPS coordinates and information from OpenStreetMap Nominatim API.
    
    Copyright (C) 2019  Diego del Pozo
    
    Thanks to Michael Bungenstock for his Teekesselchen open plugin. Checking its
    code made me try to develop a new plugin   
 
--------------------------------------------------------------------------------

ReverseGeocode.lua

Provides the logic.

------------------------------------------------------------------------------]]
ReverseGeocode ={}

local LrApplication = import "LrApplication"
local LrDialogs = import "LrDialogs"
local LrProgressScope = import "LrProgressScope"
local LrTasks = import "LrTasks"
local LrFileUtils = import "LrFileUtils"
local LrHttp = import "LrHttp"
local LrXml = import "LrXml"
local LrDate = import "LrDate"

local inspect = require 'inspect'

local lrVersion = LrApplication.versionTable()
local lrMajor = lrVersion.major
local lrMinor = lrVersion.minor
local supportsFlag = lrMajor >= 4

local cachedResult = {}
setmetatable(cachedResult, {__mode = "v"})  -- make values weak


local function httpQueryNominatim(query,debug)
	local result = {}
	local XMLDom	
	local XMLDomChild
	local XMLDomSubChild
	result["ISOCountry"]=""
	result["Country"]=""
	result["State"]=""
	result["City"]=""
	result["Location"]=""	

	local ISOCountry
	local Country
	local State
	local City
	local Location


	local myHeaders = {
		field = 'User-Agent', value = "ReverseGeocode/0.1 LightroomPLugin/8.0"
	}
	
	if debug then
		logger:debugf("Query: %s",query)
	end
	
	local body, headers = LrHttp.get(query,myHeaders,5)
	
	
	if debug then
		logger:debugf("Query: %s",query)
		logger:debugf("Answer:%s",body)
		logger:debugf("Headers:%s",inspect(headers))		
	end
	
	if headers.status == 200 then		
	
		XMLDom = LrXml.parseXml(body)
	
		for i = 1, XMLDom:childCount() do
			if XMLDom:childAtIndex(i):name() == "addressparts" then
				for j = 1, XMLDom:childAtIndex(i):childCount() do
					if XMLDom:childAtIndex(i):childAtIndex(j):name() == "country_code" then	
						result["ISOCountry"]=XMLDom:childAtIndex(i):childAtIndex(j):text()
					end
				
					if XMLDom:childAtIndex(i):childAtIndex(j):name() == "country" then	
						result["Country"]=XMLDom:childAtIndex(i):childAtIndex(j):text()
					end
				
					if XMLDom:childAtIndex(i):childAtIndex(j):name() == "state" or  XMLDom:childAtIndex(i):childAtIndex(j):name() == "county" or XMLDom:childAtIndex(i):childAtIndex(j):name() == "province"  then	
						result["State"]=XMLDom:childAtIndex(i):childAtIndex(j):text()
					end
				
					if XMLDom:childAtIndex(i):childAtIndex(j):name() == "hamlet" or XMLDom:childAtIndex(i):childAtIndex(j):name() == "village" or XMLDom:childAtIndex(i):childAtIndex(j):name() == "town" or XMLDom:childAtIndex(i):childAtIndex(j):name() == "city"   then	
						result["City"]=XMLDom:childAtIndex(i):childAtIndex(j):text()
					end
				end
			end
			if XMLDom:childAtIndex(i):name() == "namedetails" then
				for j = 1, XMLDom:childAtIndex(i):childCount() do
					if debug then
						logger:debugf("Namedetails variable: %s",XMLDom:childAtIndex(i):childAtIndex(j):name())
						logger:debugf("Namedetails value: %s",XMLDom:childAtIndex(i):childAtIndex(j):text())
					end
					if XMLDom:childAtIndex(i):childAtIndex(j):name() == "name" then	
						result["Location"]=XMLDom:childAtIndex(i):childAtIndex(j):text()
						break
					end
				end
			end
		end
	
		if debug then
			logger:debugf("Text: %s",XMLDom:text())
			logger:debugf("Name: %s",XMLDom:name())
		end
		return result

	else
		logger:errorf("Error calling nominatim: %s",query)
		logger:errorf("Response: %s",body)
		logger:errorf("Headers: %s",inspect(headers))
		result["ERROR"]=true
		result["ERRORSTRING"]=headers.error.name
		result["ERRORCODE"]=headers.error.errorcode
		logger:errorf("ERROR:%s %s %s",result["ERROR"],result["ERRORCODE"],result["ERRORSTRING"])
		return result
	end
end

local function callNominatim(settings, GPSCoordinates)
	local result = {}
	result["ISOCountry"]=""
	result["Country"]=""
	result["State"]=""
	result["City"]=""
	result["Location"]=""
	result["ERROR"]=false
	result["ERRORSTRING"]=""
	result["ERRORCODE"]=""

	local resultQuery = {}
	local logger = _G.logger
 	local doLog = settings.activateLogging	
	local coordKey = GPSCoordinates["latitude"] .. "-" .. GPSCoordinates["longitude"]
	local query
	local queryZoom
	local checkStandard = settings.settingISOCountry or settings.settingCountry or settings.settingState or settings.settingCity
	local checkLocation = settings.settingLocation
	local waitTime
	
	if cachedResult[coordKey] then
		if doLog then
			logger:debugf("Cached GPS Coordinates Lat=%s,Lon=%s",GPSCoordinates["latitude"],GPSCoordinates["longitude"])
		end
		return cachedResult[coordKey]
	else 
		--do something
		query = settings.settingNominatimHost .. "reverse?" .. "format=xml" .. "&" .. "lat=" .. GPSCoordinates["latitude"] .. "&" .. "lon=" .. GPSCoordinates["longitude"] .."&" .. "accept-language=" .. settings.settingLanguage .. "&namedetails=1" 
		
		if settings.settingEmail and settings.settingEmail ~="" then
			query = query .. "&email=" .. settings.settingEmail
		end
		

		if checkStandard then 
			queryZoom = query .. "&zoom=" .. settings.zoomCity
			waitTime = settings.settingNominatimTime - (LrDate.currentTime() - settings.processingtime)
			if waitTime > 0 then
					if doLog then
						logger:debugf("Waiting %s seconds",waitTime)
					end
				LrTasks.sleep(waitTime )
			end
			resultQuery = httpQueryNominatim (queryZoom,doLog)
			
			if resultQuery["ERRORCODE"] == "timedOut" then
				--try again once if timedOut--
				LrTasks.sleep(2)
				resultQuery = httpQueryNominatim (queryZoom,doLog)
			end

			if resultQuery["ERROR"] == true then
				result["ERROR"]=true
				result["ERRORSTRING"]=resultQuery["ERRORSTRING"]
				return result
			end
			settings.processingtime = LrDate.currentTime()
			result["ISOCountry"] = resultQuery["ISOCountry"]
			result["Country"] = resultQuery["Country"]
			result["State"] = resultQuery["State"]
			result["City"] = resultQuery["City"]
		end
		
	
		if checkLocation then 
			waitTime = settings.settingNominatimTime - (LrDate.currentTime() - settings.processingtime)
			if waitTime > 0 then
					if doLog then
						logger:debugf("Waiting %s seconds",waitTime)
					end
				LrTasks.sleep(waitTime)
			end
			queryZoom = query .. "&zoom=" .. settings.zoomLocation
			resultQuery = httpQueryNominatim (queryZoom,doLog)
			if resultQuery["ERRORCODE"] == "timedOut" then
				--try again once if timedOut--
				LrTasks.sleep(2)
				resultQuery = httpQueryNominatim (queryZoom,doLog)
			end
			if resultQuery["ERROR"] == true then
				result["ERROR"]=true
				result["ERRORSTRING"]=resultQuery["ERRORSTRING"]
				return result
			end
			settings.processingtime = LrDate.currentTime()
			result["Location"] = resultQuery["Location"]
		end

		if doLog then
			logger:debugf("Not cached GPS Coordinates Lat=%s,Lon=%s",GPSCoordinates["latitude"],GPSCoordinates["longitude"])
			logger:debugf("Query result: %s/%s/%s/%s/%s", result["ISOCountry"], result["Country"], result["State"], result["City"], result["Location"])			
		end
		
		cachedResult[coordKey]=result
		
		return result		
	end
end

function ReverseGeocode.new(context)
	local self = {}
	local catalog = LrApplication.activeCatalog()
	local photos = catalog:getMultipleSelectedOrAllPhotos()

	self.total = #photos
	self.skipped = 0 --NO GPS information or already tagged
	self.found = 0 --with information
	self.unknown = 0 --unknown location
	self.error = false
	self.errorString = ""


	function self.check_numberValue(view,value)
		local num = tonumber(value)
		if num == nil then
			return false, value, value .. " is not a number. Please enter a valid number, e.g. 3"
		end
		return true, value
	end

	function self.hasWriteAccess()
		return catalog.hasWriteAccess
	end

	function self.GPSInfo(settings)
  		local logger = _G.logger
  		local doLog = settings.activateLogging
	  	local photo
		local GPSCoordinates ={}
		local result = {}
		local doISOCountry = settings.settingISOCountry
		local doCountry = settings.settingCountry
		local doState = settings.settingState
		local doCity = settings.settingCity
		local doLocation = settings.settingLocation		
		local doOverwrite = settings.settingOverwrite
		local photoISOCountry
        local photoCountry
        local photoState
        local photoCity
        local photoLocation
		local worthChecking
		local isPhotoISOCountry
        local isPhotoCountry
        local isPhotoState
        local isPhotoCity
        local isPhotoLocation
		local foundCounter = 0
		local skipCounter = 0
		local unknownCounter = 0
  		local writingOperations


  		if doLog then
  			logger:debug("GPSInfo")
  		end


--  		When polished move to this one
  		local progressScope = LrProgressScope( {
		  		title = LOC "$$$/ReverseGeocode/ProcessingTitle=Gather GPS information",
		  		functionContext = context, 
		  		} )
		progressScope.setCancelable = true
		progressScope.setPausable = true
		
		--Temporary blocking app for easier debugging
--		local progressScope = LrDialogs.showModalProgressDialog({
--				title = LOC "$$$/ReverseGeocode/ProcessingTitle=Gather GPS information",
--				functionContext = context, 
--		} )
		
		settings.processingtime = LrDate.currentTime()
		
  		catalog:withWriteAccessDo("$$$/ReverseGeocode/PluginName=Reverse Geocode", function()
  			for i=1,self.total do
  				-- do the interface stuff at the beginning
  				if progressScope:isCanceled() then
	  				break
  				end
  				progressScope:setPortionComplete(i, self.total)

  				-- select the current photo
  				photo = photos[i]

  				progressScope:setCaption(LOC("$$$/ReverseGeocode/Processing=Processing photo #^1 out of ^2", i, self.total).. " " .. photo:getFormattedMetadata("fileName"))
 	 			LrTasks.yield()


				GPSCoordinates =  photo:getRawMetadata("gps")
				photoISOCountry =  photo:getFormattedMetadata("isoCountryCode")
				photoCountry =  photo:getFormattedMetadata("country")
				photoState =  photo:getFormattedMetadata("stateProvince")
				photoCity =  photo:getFormattedMetadata("city")
				photoLocation =  photo:getFormattedMetadata("location")
				writingOperations = false

  				if doLog then
  					logger:debugf("Processing photo %s (#%i)", photo:getFormattedMetadata("fileName"), i)
  					logger:debugf("Current data %s/%s/%s/%s/%s", photoISOCountry, photoCountry, photoState, photoCity, photoLocation)  					
  				end
				
				if GPSCoordinates ~= nil then
					isPhotoISOCountry = doISOCountry and (photoISOCountry == nil or photoISOCountry =="")
					isPhotoCountry = doCountry and (photoCountry == nil or photoCountry =="")
					isPhotoState = doState and (photoState == nil or photoState =="")
					isPhotoCity = doCity and (photoState == nil or photoState =="")
					isPhotoLocation = doLocation and (photoLocation == nil or photoLocation =="")
					worthChecking = doOverwrite or isPhotoISOCountry or isPhotoCountry or isPhotoState or isPhotoCity or isPhotoLocation
				
--					logger:debugf("MyVariables: \n \t isPhotoISOCountry = %s \n \t doISOCountry = %s \n \t photoISOCountry = %s \n \t isPhotoCountry = %s \n \t doCountry = %s \n \t photoCountry = %s \n \t isPhotoState = %s \n \t doState = %s \n \t photoState = %s \n \t isPhotoCity = %s \n \t doCity  = %s \n \t photoState = %s \n \t isPhotoLocation = %s \n \t doLocation = %s \n \t photoLocation = %s \n \t", isPhotoISOCountry, doISOCountry, photoISOCountry, isPhotoCountry, doCountry, photoCountry, isPhotoState, doState, photoState, isPhotoCity, doCity, photoState, isPhotoLocation, doLocation, photoLocation)
				
				
					if doLog then
						logger:debugf("Checking: Total %s, ISO: %s, Country: %s, State: %s, City: %s, Location: %s",worthChecking,isPhotoISOCountry,isPhotoCountry,isPhotoState,isPhotoCity,isPhotoLocation)
					end
					
					if worthChecking then
					
						if doLog then
							logger:debugf("Calling Nominatim at %s", LrDate.timeToUserFormat(LrDate.currentTime(),"%1H:%M:%S",false))
						end
					
						result=callNominatim(settings, GPSCoordinates)
						if result["ERROR"] == true then
							self.error=true
							self.errorString = LOC "$$$/ReverseGeocode/ErrorHTTP=Error calling nominatim, check console log for more information"
							self.errorString =self.errorString .. "\n\n" .. result["ERRORSTRING"] 
							return self
						end
					
		
					
						if doISOCountry and (isPhotoISOCountry or doOverwrite) and result["ISOCountry"] ~= "" then
							if doLog then
							logger:debugf("Setting ISO Country to: %s",result["ISOCountry"])
							end
							photo:setRawMetadata ("isoCountryCode", result["ISOCountry"])
							writingOperations = true
						end
					
						if doCountry and (isPhotoCountry or doOverwrite) and result["Country"] ~= "" then
							if doLog then
							logger:debugf("Setting Country to: %s",result["Country"])
							end
							photo:setRawMetadata ("country", result["Country"])
							writingOperations = true
						end
					
						if doState and (isPhotoState or doOverwrite) and result["State"] ~= "" then
							if doLog then
							logger:debugf("Setting State to: %s",result["State"])
							end
							photo:setRawMetadata ("stateProvince", result["State"])
							writingOperations = true
						end
					
						if doCity and (isPhotoCity or doOverwrite) and result["City"] ~= "" then
							if doLog then
							logger:debugf("Setting City to: %s",result["City"])
							end
							photo:setRawMetadata ("city", result["City"])
							writingOperations = true
						end
					
						if doLocation and (isPhotoLocation or doOverwrite) and result["Location"] ~= "" then
							if doLog then
							logger:debugf("Setting Location to: %s",result["Location"])
							end
							photo:setRawMetadata ("location", result["Location"])
							writingOperations = true
						end

						if writingOperations then
							foundCounter = foundCounter + 1
						else
							unknownCounter = unknownCounter + 1
						end

					else
						skipCounter = skipCounter + 1
					end

				else 
					if doLog then 
						logger:debug("No Coordinates")
					end
					skipCounter = skipCounter + 1
				end
		end
					
end)

		progressScope:done()
		self.found = foundCounter
		self.skipped = skipCounter
		self.unknown = unknownCounter
	end
	

	return self
end

